(function(){
  const SAJiL = window.SAJIL = window.SAJIL || {};
  SAJiL.features = SAJiL.features || {};
  const F = SAJiL.features.dashboardMini = {};

  function dayKey(d){
    return (SAJiL.util?.dayKey ? SAJiL.util.dayKey(d) : d.toISOString().slice(0,10));
  }

  function sumDay(key){
    const all = SAJiL.storage.getTx();
    let inc=0, exp=0;
    all.forEach(t=>{
      if(!t || !t.date) return;
      if(t.currency !== '₪') return;      // Dashboard A على ₪ فقط
      if((t.date||'').slice(0,10) !== key) return;
      if(t.type === 'income') inc += (+t.amount||0);
      else exp += (+t.amount||0);
    });
    return {inc, exp};
  }

  function last7(){
    const out=[];
    const now=new Date();
    for(let i=6;i>=0;i--){
      const d=new Date(now); d.setDate(d.getDate()-i);
      const k=dayKey(d);
      out.push({key:k, ...sumDay(k)});
    }
    return out;
  }

  function drawDonut(canvas, income, expense){
    const ctx = canvas.getContext('2d');
    const W=canvas.width, H=canvas.height;
    ctx.clearRect(0,0,W,H);

    const total = Math.max(1, income+expense);
    const cx=W/2, cy=H/2;
    const r=Math.min(W,H)*0.40;
    const lw=Math.max(14, r*0.45);

    ctx.lineWidth = lw;
    ctx.lineCap = 'round';

    // bg ring
    ctx.strokeStyle='rgba(14,27,61,.08)';
    ctx.beginPath(); ctx.arc(cx,cy,r,0,Math.PI*2); ctx.stroke();

    // expense arc
    const expA = (expense/total)*Math.PI*2;
    ctx.strokeStyle='rgba(255,77,109,.92)';
    ctx.beginPath(); ctx.arc(cx,cy,r,-Math.PI/2,-Math.PI/2+expA); ctx.stroke();

    // income arc (after expense)
    const incA = (income/total)*Math.PI*2;
    ctx.strokeStyle='rgba(25,195,125,.92)';
    ctx.beginPath(); ctx.arc(cx,cy,r,-Math.PI/2+expA,-Math.PI/2+expA+incA); ctx.stroke();

    // center text
    const net = Math.round(income-expense);
    ctx.textAlign='center'; ctx.textBaseline='middle';
    ctx.fillStyle='rgba(14,27,61,.90)';
    ctx.font='900 22px Cairo, system-ui';
    ctx.fillText(`${net}₪`, cx, cy-6);
    ctx.fillStyle='rgba(14,27,61,.55)';
    ctx.font='900 12px Cairo, system-ui';
    ctx.fillText('صافي 7 أيام', cx, cy+18);
  }

  function drawBars(canvas, days){
    const ctx=canvas.getContext('2d');
    const parent = canvas.parentElement;
    const W = Math.max(260, parent.clientWidth);
    if(canvas.width !== W) canvas.width = W;

    const H = canvas.height;
    ctx.clearRect(0,0,W,H);

    const maxV = Math.max(1, ...days.map(d=>Math.max(d.inc,d.exp)));
    const pad=10;
    const colW = (W - pad*2)/days.length;
    const base = H - 18;

    ctx.font='900 10px Cairo, system-ui';
    ctx.textAlign='center';

    days.forEach((d,i)=>{
      const x = pad + i*colW + colW/2;
      const hInc = (d.inc/maxV)*(H-30);
      const hExp = (d.exp/maxV)*(H-30);

      // expense (red)
      ctx.fillStyle='rgba(255,77,109,.55)';
      ctx.fillRect(x-colW*0.18, base-hExp, colW*0.16, hExp);

      // income (green)
      ctx.fillStyle='rgba(25,195,125,.55)';
      ctx.fillRect(x+colW*0.02, base-hInc, colW*0.16, hInc);

      // label dd
      const dd = d.key.slice(8,10);
      ctx.fillStyle='rgba(14,27,61,.55)';
      ctx.fillText(dd, x, H-6);
    });
  }

  F.render = function(){
    const donut = document.getElementById('dashDonut');
    const bars  = document.getElementById('dashBars');
    const hint  = document.getElementById('dashMiniHint');
    if(!donut || !bars) return;

    const days = last7();
    const totalInc = Math.round(days.reduce((a,x)=>a+x.inc,0));
    const totalExp = Math.round(days.reduce((a,x)=>a+x.exp,0));

    drawDonut(donut, totalInc, totalExp);
    drawBars(bars, days);

    if(hint){
      hint.textContent = `⬆︎ ${totalInc}₪ • ⬇︎ ${totalExp}₪ • ${days[0].key} → ${days[6].key}`;
    }
  };

  F.init = function(){
    window.addEventListener('resize', ()=>{
      clearTimeout(F._t);
      F._t=setTimeout(F.render, 120);
    });
  };
})();
